import { CommonModule } from '@angular/common';
import { Component, EventEmitter, Host, Inject, InjectionToken, Input, NgModule, Optional, Output, Pipe, Renderer2, SkipSelf, ViewChild, forwardRef } from '@angular/core';
import { ControlContainer, NG_VALUE_ACCESSOR, ReactiveFormsModule } from '@angular/forms';

class SignaturePadCardGroupComponent {
    constructor() {
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.control = (this.group.get(this.config.name));
    }
    /**
     * @return {?}
     */
    clear() {
        this.signaturePadControl.clear();
    }
}
SignaturePadCardGroupComponent.decorators = [
    { type: Component, args: [{
                selector: 'signature-pad-card-group',
                template: `
    <div class="signature-pad-card" [ngClass]="theme" [formGroup]="group">

      <div class="signature-pad">
        <signature-pad-control [formControlName]="config.name" #signaturePad>
        </signature-pad-control>
      </div>
      <div class="line-break"></div>
      <div class="actions">
        <button class="clear" (click)="clear()">
          {{'Clear Signature'}}
        </button>
      </div>

      <div class="feedback">

        <!-- initial message -->
        <div *ngIf="control?.untouched && control?.pristine">Sign above!</div>

        <!-- error messages -->
        <div *ngIf="control?.dirty && control?.invalid">
          <span class="form-text text-danger" *ngFor="let error of (control?.errors | objToArray)">
           {{error.value.message}}
          </span>
        </div>

        <!-- valid message -->
        <div *ngIf="control?.dirty && control.valid">Thank you!</div>

      </div>
    </div>
  `,
                providers: []
            },] },
];
/**
 * @nocollapse
 */
SignaturePadCardGroupComponent.ctorParameters = () => [];
SignaturePadCardGroupComponent.propDecorators = {
    'theme': [{ type: Input },],
    'group': [{ type: Input },],
    'config': [{ type: Input },],
    'signaturePadControl': [{ type: ViewChild, args: ['signaturePad',] },],
};

const GlobalSignaturePadConfig = new InjectionToken('Global SignaturePad default config');

/*!
 * Signature Pad v2.3.2
 * https://github.com/szimek/signature_pad
 *
 * Copyright 2017 Szymon Nowak
 * Released under the MIT license
 *
 * The main idea and some parts of the code (e.g. drawing variable width Bézier curve) are taken from:
 * http://corner.squareup.com/2012/07/smoother-signatures.html
 *
 * Implementation of interpolation using cubic Bézier curves is taken from:
 * http://benknowscode.wordpress.com/2012/09/14/path-interpolation-using-cubic-bezier-and-control-point-estimation-in-javascript
 *
 * Algorithm for approximated length of a Bézier curve is taken from:
 * http://www.lemoda.net/maths/bezier-length/index.html
 *
 */

function Point(x, y, time) {
  this.x = x;
  this.y = y;
  this.time = time || new Date().getTime();
}

Point.prototype.velocityFrom = function (start) {
  return this.time !== start.time ? this.distanceTo(start) / (this.time - start.time) : 1;
};

Point.prototype.distanceTo = function (start) {
  return Math.sqrt(Math.pow(this.x - start.x, 2) + Math.pow(this.y - start.y, 2));
};

Point.prototype.equals = function (other) {
  return this.x === other.x && this.y === other.y && this.time === other.time;
};

function Bezier(startPoint, control1, control2, endPoint) {
  this.startPoint = startPoint;
  this.control1 = control1;
  this.control2 = control2;
  this.endPoint = endPoint;
}

// Returns approximated length.
Bezier.prototype.length = function () {
  var steps = 10;
  var length = 0;
  var px = void 0;
  var py = void 0;

  for (var i = 0; i <= steps; i += 1) {
    var t = i / steps;
    var cx = this._point(t, this.startPoint.x, this.control1.x, this.control2.x, this.endPoint.x);
    var cy = this._point(t, this.startPoint.y, this.control1.y, this.control2.y, this.endPoint.y);
    if (i > 0) {
      var xdiff = cx - px;
      var ydiff = cy - py;
      length += Math.sqrt(xdiff * xdiff + ydiff * ydiff);
    }
    px = cx;
    py = cy;
  }

  return length;
};

/* eslint-disable no-multi-spaces, space-in-parens */
Bezier.prototype._point = function (t, start, c1, c2, end) {
  return start * (1.0 - t) * (1.0 - t) * (1.0 - t) + 3.0 * c1 * (1.0 - t) * (1.0 - t) * t + 3.0 * c2 * (1.0 - t) * t * t + end * t * t * t;
};

/* eslint-disable */

// http://stackoverflow.com/a/27078401/815507
function throttle(func, wait, options) {
  var context, args, result;
  var timeout = null;
  var previous = 0;
  if (!options) options = {};
  var later = function later() {
    previous = options.leading === false ? 0 : Date.now();
    timeout = null;
    result = func.apply(context, args);
    if (!timeout) context = args = null;
  };
  return function () {
    var now = Date.now();
    if (!previous && options.leading === false) previous = now;
    var remaining = wait - (now - previous);
    context = this;
    args = arguments;
    if (remaining <= 0 || remaining > wait) {
      if (timeout) {
        clearTimeout(timeout);
        timeout = null;
      }
      previous = now;
      result = func.apply(context, args);
      if (!timeout) context = args = null;
    } else if (!timeout && options.trailing !== false) {
      timeout = setTimeout(later, remaining);
    }
    return result;
  };
}

function SignaturePad(canvas, options) {
  var self = this;
  var opts = options || {};

  this.velocityFilterWeight = opts.velocityFilterWeight || 0.7;
  this.minWidth = opts.minWidth || 0.5;
  this.maxWidth = opts.maxWidth || 2.5;
  this.throttle = 'throttle' in opts ? opts.throttle : 16; // in miliseconds
  this.minDistance = 'minDistance' in opts ? opts.minDistance : 5;

  if (this.throttle) {
    this._strokeMoveUpdate = throttle(SignaturePad.prototype._strokeUpdate, this.throttle);
  } else {
    this._strokeMoveUpdate = SignaturePad.prototype._strokeUpdate;
  }

  this.dotSize = opts.dotSize || function () {
    return (this.minWidth + this.maxWidth) / 2;
  };
  this.penColor = opts.penColor || 'black';
  this.backgroundColor = opts.backgroundColor || 'rgba(0,0,0,0)';
  this.onBegin = opts.onBegin;
  this.onEnd = opts.onEnd;

  this._canvas = canvas;
  this._ctx = canvas.getContext('2d');
  this.clear();

  // We need add these inline so they are available to unbind while still having
  // access to 'self' we could use _.bind but it's not worth adding a dependency.
  this._handleMouseDown = function (event) {
    if (event.which === 1) {
      self._mouseButtonDown = true;
      self._strokeBegin(event);
    }
  };

  this._handleMouseMove = function (event) {
    if (self._mouseButtonDown) {
      self._strokeMoveUpdate(event);
    }
  };

  this._handleMouseUp = function (event) {
    if (event.which === 1 && self._mouseButtonDown) {
      self._mouseButtonDown = false;
      self._strokeEnd(event);
    }
  };

  this._handleTouchStart = function (event) {
    if (event.targetTouches.length === 1) {
      var touch = event.changedTouches[0];
      self._strokeBegin(touch);
    }
  };

  this._handleTouchMove = function (event) {
    // Prevent scrolling.
    event.preventDefault();

    var touch = event.targetTouches[0];
    self._strokeMoveUpdate(touch);
  };

  this._handleTouchEnd = function (event) {
    var wasCanvasTouched = event.target === self._canvas;
    if (wasCanvasTouched) {
      event.preventDefault();
      self._strokeEnd(event);
    }
  };

  // Enable mouse and touch event handlers
  this.on();
}

// Public methods
SignaturePad.prototype.clear = function () {
  var ctx = this._ctx;
  var canvas = this._canvas;

  ctx.fillStyle = this.backgroundColor;
  ctx.clearRect(0, 0, canvas.width, canvas.height);
  ctx.fillRect(0, 0, canvas.width, canvas.height);

  this._data = [];
  this._reset();
  this._isEmpty = true;
};

SignaturePad.prototype.fromDataURL = function (dataUrl) {
  var _this = this;

  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

  var image = new Image();
  var ratio = options.ratio || window.devicePixelRatio || 1;
  var width = options.width || this._canvas.width / ratio;
  var height = options.height || this._canvas.height / ratio;

  this._reset();
  image.src = dataUrl;
  image.onload = function () {
    _this._ctx.drawImage(image, 0, 0, width, height);
  };
  this._isEmpty = false;
};

SignaturePad.prototype.toDataURL = function (type) {
  var _canvas;

  switch (type) {
    case 'image/svg+xml':
      return this._toSVG();
    default:
      for (var _len = arguments.length, options = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
        options[_key - 1] = arguments[_key];
      }

      return (_canvas = this._canvas).toDataURL.apply(_canvas, [type].concat(options));
  }
};

SignaturePad.prototype.on = function () {
  this._handleMouseEvents();
  this._handleTouchEvents();
};

SignaturePad.prototype.off = function () {
  this._canvas.removeEventListener('mousedown', this._handleMouseDown);
  this._canvas.removeEventListener('mousemove', this._handleMouseMove);
  document.removeEventListener('mouseup', this._handleMouseUp);

  this._canvas.removeEventListener('touchstart', this._handleTouchStart);
  this._canvas.removeEventListener('touchmove', this._handleTouchMove);
  this._canvas.removeEventListener('touchend', this._handleTouchEnd);
};

SignaturePad.prototype.isEmpty = function () {
  return this._isEmpty;
};

// Private methods
SignaturePad.prototype._strokeBegin = function (event) {
  this._data.push([]);
  this._reset();
  this._strokeUpdate(event);

  if (typeof this.onBegin === 'function') {
    this.onBegin(event);
  }
};

SignaturePad.prototype._strokeUpdate = function (event) {
  var x = event.clientX;
  var y = event.clientY;

  var point = this._createPoint(x, y);
  var lastPointGroup = this._data[this._data.length - 1];
  var lastPoint = lastPointGroup && lastPointGroup[lastPointGroup.length - 1];
  var isLastPointTooClose = lastPoint && point.distanceTo(lastPoint) < this.minDistance;

  // Skip this point if it's too close to the previous one
  if (!(lastPoint && isLastPointTooClose)) {
    var _addPoint = this._addPoint(point),
        curve = _addPoint.curve,
        widths = _addPoint.widths;

    if (curve && widths) {
      this._drawCurve(curve, widths.start, widths.end);
    }

    this._data[this._data.length - 1].push({
      x: point.x,
      y: point.y,
      time: point.time,
      color: this.penColor
    });
  }
};

SignaturePad.prototype._strokeEnd = function (event) {
  var canDrawCurve = this.points.length > 2;
  var point = this.points[0]; // Point instance

  if (!canDrawCurve && point) {
    this._drawDot(point);
  }

  if (point) {
    var lastPointGroup = this._data[this._data.length - 1];
    var lastPoint = lastPointGroup[lastPointGroup.length - 1]; // plain object

    // When drawing a dot, there's only one point in a group, so without this check
    // such group would end up with exactly the same 2 points.
    if (!point.equals(lastPoint)) {
      lastPointGroup.push({
        x: point.x,
        y: point.y,
        time: point.time,
        color: this.penColor
      });
    }
  }

  if (typeof this.onEnd === 'function') {
    this.onEnd(event);
  }
};

SignaturePad.prototype._handleMouseEvents = function () {
  this._mouseButtonDown = false;

  this._canvas.addEventListener('mousedown', this._handleMouseDown);
  this._canvas.addEventListener('mousemove', this._handleMouseMove);
  document.addEventListener('mouseup', this._handleMouseUp);
};

SignaturePad.prototype._handleTouchEvents = function () {
  // Pass touch events to canvas element on mobile IE11 and Edge.
  this._canvas.style.msTouchAction = 'none';
  this._canvas.style.touchAction = 'none';

  this._canvas.addEventListener('touchstart', this._handleTouchStart);
  this._canvas.addEventListener('touchmove', this._handleTouchMove);
  this._canvas.addEventListener('touchend', this._handleTouchEnd);
};

SignaturePad.prototype._reset = function () {
  this.points = [];
  this._lastVelocity = 0;
  this._lastWidth = (this.minWidth + this.maxWidth) / 2;
  this._ctx.fillStyle = this.penColor;
};

SignaturePad.prototype._createPoint = function (x, y, time) {
  var rect = this._canvas.getBoundingClientRect();

  return new Point(x - rect.left, y - rect.top, time || new Date().getTime());
};

SignaturePad.prototype._addPoint = function (point) {
  var points = this.points;
  var tmp = void 0;

  points.push(point);

  if (points.length > 2) {
    // To reduce the initial lag make it work with 3 points
    // by copying the first point to the beginning.
    if (points.length === 3) points.unshift(points[0]);

    tmp = this._calculateCurveControlPoints(points[0], points[1], points[2]);
    var c2 = tmp.c2;
    tmp = this._calculateCurveControlPoints(points[1], points[2], points[3]);
    var c3 = tmp.c1;
    var curve = new Bezier(points[1], c2, c3, points[2]);
    var widths = this._calculateCurveWidths(curve);

    // Remove the first element from the list,
    // so that we always have no more than 4 points in points array.
    points.shift();

    return { curve: curve, widths: widths };
  }

  return {};
};

SignaturePad.prototype._calculateCurveControlPoints = function (s1, s2, s3) {
  var dx1 = s1.x - s2.x;
  var dy1 = s1.y - s2.y;
  var dx2 = s2.x - s3.x;
  var dy2 = s2.y - s3.y;

  var m1 = { x: (s1.x + s2.x) / 2.0, y: (s1.y + s2.y) / 2.0 };
  var m2 = { x: (s2.x + s3.x) / 2.0, y: (s2.y + s3.y) / 2.0 };

  var l1 = Math.sqrt(dx1 * dx1 + dy1 * dy1);
  var l2 = Math.sqrt(dx2 * dx2 + dy2 * dy2);

  var dxm = m1.x - m2.x;
  var dym = m1.y - m2.y;

  var k = l2 / (l1 + l2);
  var cm = { x: m2.x + dxm * k, y: m2.y + dym * k };

  var tx = s2.x - cm.x;
  var ty = s2.y - cm.y;

  return {
    c1: new Point(m1.x + tx, m1.y + ty),
    c2: new Point(m2.x + tx, m2.y + ty)
  };
};

SignaturePad.prototype._calculateCurveWidths = function (curve) {
  var startPoint = curve.startPoint;
  var endPoint = curve.endPoint;
  var widths = { start: null, end: null };

  var velocity = this.velocityFilterWeight * endPoint.velocityFrom(startPoint) + (1 - this.velocityFilterWeight) * this._lastVelocity;

  var newWidth = this._strokeWidth(velocity);

  widths.start = this._lastWidth;
  widths.end = newWidth;

  this._lastVelocity = velocity;
  this._lastWidth = newWidth;

  return widths;
};

SignaturePad.prototype._strokeWidth = function (velocity) {
  return Math.max(this.maxWidth / (velocity + 1), this.minWidth);
};

SignaturePad.prototype._drawPoint = function (x, y, size) {
  var ctx = this._ctx;

  ctx.moveTo(x, y);
  ctx.arc(x, y, size, 0, 2 * Math.PI, false);
  this._isEmpty = false;
};

SignaturePad.prototype._drawCurve = function (curve, startWidth, endWidth) {
  var ctx = this._ctx;
  var widthDelta = endWidth - startWidth;
  var drawSteps = Math.floor(curve.length());

  ctx.beginPath();

  for (var i = 0; i < drawSteps; i += 1) {
    // Calculate the Bezier (x, y) coordinate for this step.
    var t = i / drawSteps;
    var tt = t * t;
    var ttt = tt * t;
    var u = 1 - t;
    var uu = u * u;
    var uuu = uu * u;

    var x = uuu * curve.startPoint.x;
    x += 3 * uu * t * curve.control1.x;
    x += 3 * u * tt * curve.control2.x;
    x += ttt * curve.endPoint.x;

    var y = uuu * curve.startPoint.y;
    y += 3 * uu * t * curve.control1.y;
    y += 3 * u * tt * curve.control2.y;
    y += ttt * curve.endPoint.y;

    var width = startWidth + ttt * widthDelta;
    this._drawPoint(x, y, width);
  }

  ctx.closePath();
  ctx.fill();
};

SignaturePad.prototype._drawDot = function (point) {
  var ctx = this._ctx;
  var width = typeof this.dotSize === 'function' ? this.dotSize() : this.dotSize;

  ctx.beginPath();
  this._drawPoint(point.x, point.y, width);
  ctx.closePath();
  ctx.fill();
};

SignaturePad.prototype._fromData = function (pointGroups, drawCurve, drawDot) {
  for (var i = 0; i < pointGroups.length; i += 1) {
    var group = pointGroups[i];

    if (group.length > 1) {
      for (var j = 0; j < group.length; j += 1) {
        var rawPoint = group[j];
        var point = new Point(rawPoint.x, rawPoint.y, rawPoint.time);
        var color = rawPoint.color;

        if (j === 0) {
          // First point in a group. Nothing to draw yet.

          // All points in the group have the same color, so it's enough to set
          // penColor just at the beginning.
          this.penColor = color;
          this._reset();

          this._addPoint(point);
        } else if (j !== group.length - 1) {
          // Middle point in a group.
          var _addPoint2 = this._addPoint(point),
              curve = _addPoint2.curve,
              widths = _addPoint2.widths;

          if (curve && widths) {
            drawCurve(curve, widths, color);
          }
        } else {
          // Last point in a group. Do nothing.
        }
      }
    } else {
      this._reset();
      var _rawPoint = group[0];
      drawDot(_rawPoint);
    }
  }
};

SignaturePad.prototype._toSVG = function () {
  var _this2 = this;

  var pointGroups = this._data;
  var canvas = this._canvas;
  var ratio = Math.max(window.devicePixelRatio || 1, 1);
  var minX = 0;
  var minY = 0;
  var maxX = canvas.width / ratio;
  var maxY = canvas.height / ratio;
  var svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');

  svg.setAttributeNS(null, 'width', canvas.width);
  svg.setAttributeNS(null, 'height', canvas.height);

  this._fromData(pointGroups, function (curve, widths, color) {
    var path = document.createElement('path');

    // Need to check curve for NaN values, these pop up when drawing
    // lines on the canvas that are not continuous. E.g. Sharp corners
    // or stopping mid-stroke and than continuing without lifting mouse.
    if (!isNaN(curve.control1.x) && !isNaN(curve.control1.y) && !isNaN(curve.control2.x) && !isNaN(curve.control2.y)) {
      var attr = 'M ' + curve.startPoint.x.toFixed(3) + ',' + curve.startPoint.y.toFixed(3) + ' ' + ('C ' + curve.control1.x.toFixed(3) + ',' + curve.control1.y.toFixed(3) + ' ') + (curve.control2.x.toFixed(3) + ',' + curve.control2.y.toFixed(3) + ' ') + (curve.endPoint.x.toFixed(3) + ',' + curve.endPoint.y.toFixed(3));

      path.setAttribute('d', attr);
      path.setAttribute('stroke-width', (widths.end * 2.25).toFixed(3));
      path.setAttribute('stroke', color);
      path.setAttribute('fill', 'none');
      path.setAttribute('stroke-linecap', 'round');

      svg.appendChild(path);
    }
  }, function (rawPoint) {
    var circle = document.createElement('circle');
    var dotSize = typeof _this2.dotSize === 'function' ? _this2.dotSize() : _this2.dotSize;
    circle.setAttribute('r', dotSize);
    circle.setAttribute('cx', rawPoint.x);
    circle.setAttribute('cy', rawPoint.y);
    circle.setAttribute('fill', rawPoint.color);

    svg.appendChild(circle);
  });

  var prefix = 'data:image/svg+xml;base64,';
  var header = '<svg' + ' xmlns="http://www.w3.org/2000/svg"' + ' xmlns:xlink="http://www.w3.org/1999/xlink"' + (' viewBox="' + minX + ' ' + minY + ' ' + maxX + ' ' + maxY + '"') + (' width="' + maxX + '"') + (' height="' + maxY + '"') + '>';
  var body = svg.innerHTML;

  // IE hack for missing innerHTML property on SVGElement
  if (body === undefined) {
    var dummy = document.createElement('dummy');
    var nodes = svg.childNodes;
    dummy.innerHTML = '';

    for (var i = 0; i < nodes.length; i += 1) {
      dummy.appendChild(nodes[i].cloneNode(true));
    }

    body = dummy.innerHTML;
  }

  var footer = '</svg>';
  var data = header + body + footer;

  return prefix + btoa(data);
};

SignaturePad.prototype.fromData = function (pointGroups) {
  var _this3 = this;

  this.clear();

  this._fromData(pointGroups, function (curve, widths) {
    return _this3._drawCurve(curve, widths.start, widths.end);
  }, function (rawPoint) {
    return _this3._drawDot(rawPoint);
  });

  this._data = pointGroups;
};

SignaturePad.prototype.toData = function () {
  return this._data;
};

class SignaturePadComponent {
    /**
     * @param {?} renderer
     * @param {?=} defaultConfig
     */
    constructor(renderer, defaultConfig) {
        this.renderer = renderer;
        this.defaultConfig = defaultConfig;
        this.value = [];
        this.valueChange = new EventEmitter();
        this.onEndWrapper = (value) => {
            // configurable onEnd
            if (this.isObject(this.config)
                && 'onEnd' in this.config
                && typeof this.config.onEnd === 'function') {
                this.config.onEnd(value);
            }
            // default onEnd function
            // sync form control
            this.handleInput(this.signaturePad.toData());
        };
        this.onBeginWrapper = (value) => {
            // configurable onBegin
            if (this.isObject(this.config)
                && 'onBegin' in this.config
                && typeof this.config.onBegin === 'function') {
                this.config.onBegin(value);
            }
            // default onBegin function here
        };
        this.resizeSignaturePad = () => {
            if (this.canvas) {
                this.resizeCanvas();
                // When the width or height of a canvas gets modified,
                // it will be automatically cleared by the browser.
                // How ever the data of the signature are still stores in the model value => this.value.
                // Because of this we have to reassign the value from this.value to the signaturePad.
                this.renderViewValue(this.value);
            }
        };
    }
    /**
     * @param {?} changes
     * @return {?}
     */
    ngOnChanges(changes) {
        if ('config' in changes) {
            this.updateConfig(changes.config.currentValue);
        }
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.initializeSignaturePad();
    }
    /**
     * @return {?}
     */
    ngAfterViewInit() {
        // listen to focus
        this.renderer.listen(this.canvas, 'focus', ($event) => {
            this.handleFocus(true);
        });
        // listen to blur
        this.renderer.listen(this.canvas, 'blur', ($event) => {
            this.handleFocus(false);
        });
        // handle resize
        this.resizeSignaturePad();
        this.renderer.listen('window', 'resize', ($event) => {
            this.resizeSignaturePad();
        });
    }
    /**
     * @return {?}
     */
    getValue() {
        return this.value;
    }
    /**
     * @param {?} signatureData
     * @return {?}
     */
    setValue(signatureData) {
        if (signatureData && signatureData.constructor === [].constructor) {
            this.value = [...signatureData];
        }
        else {
            this.value = null;
        }
        this.valueChange.next(this.value);
    }
    /**
     * @param {?} isDisabled
     * @return {?}
     */
    setDisabledState(isDisabled) {
        this.disabled = isDisabled;
        this.renderViewDisabled();
    }
    /**
     * @return {?}
     */
    clear() {
        this.signaturePad.clear();
        this.setValue(null);
    }
    /**
     * @param {?} signatureData
     * @return {?}
     */
    handleInput(signatureData) {
        this.setValue(signatureData);
    }
    /**
     * @param {?} isFocus
     * @return {?}
     */
    handleFocus(isFocus) {
        this.focus = isFocus;
        this.renderViewFocus();
    }
    /**
     * @return {?}
     */
    renderViewDisabled() {
        this.renderer.setProperty(this.canvas, 'disabled', this.disabled);
        if (this.disabled) {
            this.renderer.addClass(this.canvas, 'disabled');
        }
        else {
            this.renderer.removeClass(this.canvas, 'disabled');
        }
    }
    /**
     * @return {?}
     */
    renderViewFocus() {
        this.renderer.setProperty(this.canvas, 'focus', this.focus);
        if (this.focus) {
            this.renderer.addClass(this.canvas, 'focus');
        }
        else {
            this.renderer.removeClass(this.canvas, 'focus');
        }
    }
    /**
     * @param {?} signatureData
     * @return {?}
     */
    renderViewValue(signatureData) {
        if (signatureData && signatureData.constructor === [].constructor) {
            this.signaturePad.fromData([...signatureData]);
        }
        else {
            this.signaturePad.clear();
        }
    }
    /**
     * @param {?=} config
     * @return {?}
     */
    updateConfig(config) {
        if (!config || config.constructor !== {}.constructor || !this.signaturePad) {
            return;
        }
        if ('dotSize' in config && config.dotSize) {
            this.signaturePad.dotSize = config.dotSize;
        }
        if ('minWidth' in config && config.minWidth >= 0) {
            this.signaturePad.minWidth = config.minWidth;
            console.log('minWidth', this.signaturePad.minWidth, config.minWidth);
        }
        if ('maxWidth' in config && config.maxWidth >= 0) {
            this.signaturePad.maxWidth = config.maxWidth;
        }
        if ('throttle' in config && config.throttle >= 0) {
            this.signaturePad.throttle = config.throttle;
        }
        if ('backgroundColor' in config && config.backgroundColor) {
            this.signaturePad.backgroundColor = config.backgroundColor;
            this.resizeSignaturePad();
        }
        if ('penColor' in config && config.penColor) {
            this.signaturePad.penColor = config.penColor;
        }
        if ('velocityFilterWeight' in config && config.velocityFilterWeight >= 0) {
            this.signaturePad.velocityFilterWeight = config.velocityFilterWeight;
        }
        // callbacks are registered when creating signaturePad instance in initializeSignaturePad
        // onBegin
        // onEnd
    }
    /**
     * @return {?}
     */
    resizeCanvas() {
        // When zoomed out to less than 100%, for some very strange reason,
        // some browsers report devicePixelRatio as less than 1
        // and only part of the canvas is cleared then.
        // So we will have at least 1 as ration.
        const /** @type {?} */ ratio = 1; // @TODO fix offset on ratios > 1. => Math.max(window.devicePixelRatio || 1, 1);
        // information needed to calculate the available width and height
        const /** @type {?} */ canvasStyles = window.getComputedStyle(this.canvas, null);
        const /** @type {?} */ canvasBorder = ({});
        canvasBorder.top = parseInt(canvasStyles.borderTopWidth, null) || 0;
        canvasBorder.bottom = parseInt(canvasStyles.borderBottomWidth, null) || 0;
        canvasBorder.left = parseInt(canvasStyles.borderLeftWidth, null) || 0;
        canvasBorder.right = parseInt(canvasStyles.borderRightWidth, null) || 0;
        canvasBorder.fullHeight = canvasBorder.top + canvasBorder.bottom;
        canvasBorder.fullWidth = canvasBorder.left + canvasBorder.right;
        const /** @type {?} */ containerStyles = window.getComputedStyle(this.container, null);
        const /** @type {?} */ parentPadding = ({});
        parentPadding.top = parseInt(containerStyles.paddingTop, null) || 0;
        parentPadding.bottom = parseInt(containerStyles.paddingBottom, null) || 0;
        parentPadding.left = parseInt(containerStyles.paddingLeft, null) || 0;
        parentPadding.right = parseInt(containerStyles.paddingRight, null) || 0;
        parentPadding.fullHeight = parentPadding.top + parentPadding.bottom;
        parentPadding.fullWidth = parentPadding.left + parentPadding.right;
        const /** @type {?} */ withToSubtract = parentPadding.fullWidth + canvasBorder.fullWidth;
        const /** @type {?} */ heightToSubtract = parentPadding.fullHeight + canvasBorder.fullHeight;
        // resize canvas
        // get most right point of signature
        let /** @type {?} */ signatureWidth = 0;
        if (this.value && [].constructor === this.value.constructor) {
            signatureWidth = this.value
                .reduce((concated, arr) => concated.concat(arr), [])
                .reduce((mR, segment) => mR < segment.x ? segment.x : mR, 0);
        }
        // get most left point of signature
        let /** @type {?} */ signatureHeight = 0;
        if (this.value && [].constructor === this.value.constructor) {
            signatureHeight = this.value
                .reduce((concated, arr) => concated.concat(arr), [])
                .reduce((mL, segment) => mL < segment.y ? segment.y : mL, 0);
        }
        // calc new width and height
        const /** @type {?} */ newCanvasWidth = Math.max(this.container.clientWidth, signatureWidth);
        const /** @type {?} */ newCanvasHeight = Math.max(this.container.clientHeight, signatureHeight);
        // adopt canvas scales
        this.canvas.width = (newCanvasWidth - withToSubtract) * ratio;
        this.canvas.height = (newCanvasHeight - heightToSubtract - 6) * ratio; // @TODO find the 6px
        this.canvas.getContext('2d').scale(ratio, ratio);
        // adopt show/hide scroll of vertical canvas container
        this.container.style.overflowX = 'inherit';
        if (this.canvas.width + withToSubtract > this.container.clientWidth) {
            this.container.style.overflowX = 'scroll';
        }
        // adopt show/hide scroll of horizontal canvas container
        this.container.style.overflowY = 'inherit';
        if (this.canvas.height > this.container.clientHeight) {
            this.container.style.overflowY = 'scroll';
        }
        // When the width or height of a canvas gets modified,
        // it will be automatically cleared by the browser.
        // So we have to call signaturePad.clear() to make sure
        // that signaturePad.isEmpty() returns correct value in this case.
        this.signaturePad.clear();
    }
    /**
     * @return {?}
     */
    initializeSignaturePad() {
        this.canvas = this.canvasRef.nativeElement;
        this.container = this.canvas.parentElement;
        this.container.className = this.container.className + ' signature-pad';
        this.signaturePad = new SignaturePad(this.canvas, this.defaultConfig);
        // onBegin
        this.signaturePad.onBegin = this.onBeginWrapper;
        // onEnd
        this.signaturePad.onEnd = this.onEndWrapper;
        this.updateConfig(this.config);
    }
    /**
     * @param {?} val
     * @return {?}
     */
    isObject(val) {
        const /** @type {?} */ objectConstructor = {}.constructor;
        return val && val.constructor === objectConstructor;
    }
}
SignaturePadComponent.decorators = [
    { type: Component, args: [{
                selector: 'signature-pad',
                template: `
    <canvas
      tabindex="0"
      class="signature-pad-canvas"
      width="400"
      height="200"
      #signatureCanvas>
    </canvas>
  `
            },] },
];
/**
 * @nocollapse
 */
SignaturePadComponent.ctorParameters = () => [
    { type: Renderer2, },
    { type: undefined, decorators: [{ type: Optional }, { type: Inject, args: [GlobalSignaturePadConfig,] },] },
];
SignaturePadComponent.propDecorators = {
    'canvasRef': [{ type: ViewChild, args: ['signatureCanvas',] },],
    'valueChange': [{ type: Output },],
    'config': [{ type: Input },],
};

class SignaturePadControlComponent extends SignaturePadComponent {
    /**
     * @param {?} renderer
     * @param {?} parentFormContainer
     * @param {?=} defaultConfig
     */
    constructor(renderer, parentFormContainer, defaultConfig) {
        super(renderer, defaultConfig);
        this.parentFormContainer = parentFormContainer;
        this.onTouch = () => {
        };
        this.onModelChange = (value) => {
        };
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        super.ngOnInit();
        if (this.parentFormContainer) {
            this.control = this.parentFormContainer['form'].controls[this.formControlName];
        }
    }
    /**
     * @param {?} signatureData
     * @return {?}
     */
    writeValue(signatureData) {
        this.renderViewValue(signatureData);
    }
    /**
     * @param {?} fn
     * @return {?}
     */
    registerOnChange(fn) {
        this.onModelChange = fn;
    }
    /**
     * @param {?} fn
     * @return {?}
     */
    registerOnTouched(fn) {
        this.onTouch = fn;
    }
    /**
     *
     * @param {?} isFocus
     * @return {?}
     */
    handleFocus(isFocus) {
        super.handleFocus(isFocus);
        if (!isFocus) {
            this.onTouch();
        }
    }
    /**
     *
     * @param {?} signatureData
     * @return {?}
     */
    handleInput(signatureData) {
        super.handleInput(signatureData);
        this.onModelChange(signatureData);
    }
    /**
     * @return {?}
     */
    clear() {
        super.clear();
        this.onModelChange([]);
    }
}
SignaturePadControlComponent.decorators = [
    { type: Component, args: [{
                selector: 'signature-pad-control',
                template: `
    <canvas
      tabindex="0"
      class="signature-pad-canvas"
      width="400"
      height="200"
      #signatureCanvas>
    </canvas>
  `,
                providers: [
                    {
                        provide: NG_VALUE_ACCESSOR,
                        useExisting: forwardRef(() => SignaturePadControlComponent),
                        multi: true
                    }
                ]
            },] },
];
/**
 * @nocollapse
 */
SignaturePadControlComponent.ctorParameters = () => [
    { type: Renderer2, },
    { type: ControlContainer, decorators: [{ type: Optional }, { type: Host }, { type: SkipSelf },] },
    { type: undefined, decorators: [{ type: Optional }, { type: Inject, args: [GlobalSignaturePadConfig,] },] },
];
SignaturePadControlComponent.propDecorators = {
    'formControlName': [{ type: Input },],
};

class SignaturePadCardComponent extends SignaturePadControlComponent {
    /**
     * @param {?} renderer
     * @param {?} parentFormContainer
     * @param {?=} defaultConfig
     */
    constructor(renderer, parentFormContainer, defaultConfig) {
        super(renderer, parentFormContainer, defaultConfig);
        this.theme = 'raised';
    }
}
SignaturePadCardComponent.decorators = [
    { type: Component, args: [{
                selector: 'signature-pad-card',
                template: `
    <div class="signature-pad-card" [ngClass]="theme">

      <div class="signature-pad">
        <canvas
          tabindex="0"
          class="signature-pad-canvas"
          width="400"
          height="200"
          #signatureCanvas>
        </canvas>
      </div>
      <div class="line-break"></div>
      <div class="actions">
        <button class="clear" (click)="clear()">
          {{'Clear Signature'}}
        </button>
      </div>

      <div class="feedback">

        <!-- initial message -->
        <div *ngIf="control.untouched && control.pristine">Sign above!</div>

        <!-- error messages -->
        <div *ngIf="control.dirty && control.invalid">
          <span class="form-text text-danger" *ngFor="let error of (control.errors | objToArray)">
           {{error.value.message}}
          </span>
        </div>

        <!-- valid message -->
        <div *ngIf="control.dirty && control.valid">Thank you!</div>

      </div>
    </div>
  `,
                providers: [
                    {
                        provide: NG_VALUE_ACCESSOR,
                        useExisting: forwardRef(() => SignaturePadCardComponent),
                        multi: true
                    }
                ]
            },] },
];
/**
 * @nocollapse
 */
SignaturePadCardComponent.ctorParameters = () => [
    { type: Renderer2, },
    { type: ControlContainer, decorators: [{ type: Optional }, { type: Host }, { type: SkipSelf },] },
    { type: undefined, decorators: [{ type: Optional }, { type: Inject, args: [GlobalSignaturePadConfig,] },] },
];
SignaturePadCardComponent.propDecorators = {
    'theme': [{ type: Input },],
    'formControlName': [{ type: Input },],
};

class ObjToArrayPipe {
    /**
     * @param {?} value
     * @param {?=} args
     * @return {?}
     */
    transform(value, args) {
        const /** @type {?} */ objCtor = {}.constructor;
        if (value && value.constructor === objCtor) {
            return Object.keys(value)
                .map(key => {
                return { key, value: value[key] };
            });
        }
        return value;
    }
}
ObjToArrayPipe.decorators = [
    { type: Pipe, args: [{
                name: 'objToArray'
            },] },
];
/**
 * @nocollapse
 */
ObjToArrayPipe.ctorParameters = () => [];

/**
 * @param {?} c
 * @return {?}
 */
function isArray(c) {
    if (c.value) {
        const /** @type {?} */ error = {
            isArray: {
                message: 'signature is no array'
            }
        };
        const /** @type {?} */ arrayConstructor = [].constructor;
        if (c.value.constructor !== arrayConstructor) {
            return error;
        }
    }
    return null;
}

/**
 * @param {?} c
 * @return {?}
 */
function consistOfArrays(c) {
    if (c.value) {
        const /** @type {?} */ arrayConstructor = [].constructor;
        const /** @type {?} */ error = {
            consistOfArrays: {
                numOfItemsThatAreNoArray: 0,
                message: 'Signature don\'t consist of arrays'
            }
        };
        if (c.value.constructor !== arrayConstructor) {
            return null;
        }
        const /** @type {?} */ countOfNonArrays = c.value.filter(function (i) {
            // filter non arrays
            return i.constructor !== arrayConstructor;
        }).length;
        // return error if a non array is contained
        if (countOfNonArrays > 0) {
            error.consistOfArrays.numOfItemsThatAreNoArray = countOfNonArrays;
            return error;
        }
        return null;
    }
    return null;
}

/**
 * @param {?} c
 * @return {?}
 */
function consistOfObjects(c) {
    if (c.value) {
        const /** @type {?} */ objectConstructor = {}.constructor;
        const /** @type {?} */ arrayConstructor = [].constructor;
        const /** @type {?} */ error = {
            consistOfObjects: {
                numOfItemsThatAreNoObject: 0,
                message: 'Signature segments don\'t contain of objects'
            }
        };
        if (c.value.constructor !== arrayConstructor) {
            return null;
        }
        const /** @type {?} */ countOfNonObjects = c.value.reduce((nonObjCount, line) => {
            if (line.constructor !== arrayConstructor) {
                return false;
            }
            nonObjCount += line.filter((i) => {
                // filter non objects
                return i.constructor !== objectConstructor;
            }).length;
        }, 0);
        // return error if a non object is contained
        if (countOfNonObjects > 0) {
            error.consistOfObjects.numOfItemsThatAreNoObject = countOfNonObjects;
            return error;
        }
        return null;
    }
    return null;
}

/**
 * @param {?} minLineCount
 * @return {?}
 */
function minLines(minLineCount) {
    return (c) => {
        if (c.value) {
            const /** @type {?} */ lineCount = c.value.length;
            if (lineCount < minLineCount) {
                return {
                    minLines: {
                        lineCount: lineCount,
                        minLineCount: minLineCount,
                        message: `Signature must have a minimum line count of ${minLineCount}.`
                    }
                };
            }
            return null;
        }
        return null;
    };
}

/**
 * @param {?} minPointCount
 * @param {?} minLineCount
 * @return {?}
 */
function minPointsInEachNLines(minPointCount, minLineCount) {
    return (c) => {
        if (c.value) {
            const /** @type {?} */ arrayConstructor = [].constructor;
            if (c.value.constructor !== arrayConstructor) {
                return null;
            }
            const /** @type {?} */ lineCountWithValidPointCount = c.value.filter((i) => {
                if (i.constructor !== arrayConstructor) {
                    return true;
                }
                return i.length >= minPointCount;
            }).length;
            if (lineCountWithValidPointCount < minLineCount) {
                return {
                    minPointsInNLines: {
                        pointCount: lineCountWithValidPointCount,
                        minPointCount: minPointCount,
                        message: `Signature must have at least ${minPointCount} points in at least ${minLineCount} lines.`
                    }
                };
            }
            return null;
        }
        return null;
    };
}

class AngularSignaturePadModule {
    /**
     * @return {?}
     */
    static forRoot() {
        return {
            providers: [
                ObjToArrayPipe
            ],
            ngModule: AngularSignaturePadModule
        };
    }
}
AngularSignaturePadModule.decorators = [
    { type: NgModule, args: [{
                imports: [CommonModule, ReactiveFormsModule],
                declarations: [SignaturePadComponent, SignaturePadControlComponent, SignaturePadCardComponent, SignaturePadCardGroupComponent, ObjToArrayPipe],
                exports: [SignaturePadComponent, SignaturePadControlComponent, SignaturePadCardComponent, SignaturePadCardGroupComponent]
            },] },
];
/**
 * @nocollapse
 */
AngularSignaturePadModule.ctorParameters = () => [];

/**
 * Generated bundle index. Do not edit.
 */

export { GlobalSignaturePadConfig, isArray, consistOfArrays, consistOfObjects, minLines, minPointsInEachNLines, AngularSignaturePadModule, SignaturePadCardGroupComponent as ɵe, SignaturePadCardComponent as ɵd, SignaturePadControlComponent as ɵc, SignaturePadComponent as ɵa, ObjToArrayPipe as ɵf };
//# sourceMappingURL=angular-signature-pad.js.map
